#define FD_SETSIZE		1024

#include "defaultsocketregistry.h"
#include "icqlog.h"
#include <assert.h>


DefaultSocketRegistry::DefaultSocketRegistry(int msecs)
{
	timeout.tv_sec = msecs / 1000;
	timeout.tv_usec = (msecs % 1000) * 1000;
}

DefaultSocketRegistry::~DefaultSocketRegistry()
{
	map<int, FD_ENTRY *>::iterator it;
	for (it = socketMap.begin(); it != socketMap.end(); ++it)
		delete (*it).second;
}

void DefaultSocketRegistry::add(int fd, SocketListener *l)
{
	map<int, FD_ENTRY *>::iterator it = socketMap.find(fd);

	assert(it == socketMap.end());

	FD_ENTRY *e = new FD_ENTRY;
	e->events = 0;
	e->listener = l;
	socketMap[fd] = e;
}

void DefaultSocketRegistry::remove(int fd)
{
	map<int, FD_ENTRY *>::iterator it = socketMap.find(fd);

	assert(it != socketMap.end());

	delete (*it).second;
	socketMap.erase(it);
}

void DefaultSocketRegistry::selectEvent(int fd, int event)
{
	map<int, FD_ENTRY *>::iterator it = socketMap.find(fd);

	assert(it != socketMap.end());

	FD_ENTRY *e = (*it).second;
	e->events = event;
}

bool DefaultSocketRegistry::poll()
{
	fd_set readfds, writefds, exceptfds;
	int maxfd = -1;

	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	FD_ZERO(&exceptfds);

	map<int, FD_ENTRY *>::iterator it, next;

	for (it = socketMap.begin(); it != socketMap.end(); ++it) {
		int fd = (*it).first;
		FD_ENTRY *e = (*it).second;

		if (e->events & Socket::READ)
			FD_SET(fd, &readfds);
		if (e->events & Socket::WRITE)
			FD_SET(fd, &writefds);
		if (e->events & Socket::EXCEPTION)
			FD_SET(fd, &exceptfds);

		if (fd > maxfd)
			maxfd = fd;
	}

#ifdef _WIN32
	if (maxfd < 0) {
		Sleep(1000);
		return true;
	}
#endif

	timeval tv = timeout;

	int n = select(maxfd + 1, &readfds, &writefds, &exceptfds, &tv);
	if (n < 0) {
		ICQ_LOG("select() failed: \n");
		return false;
	}

	for (it = socketMap.begin(); it != socketMap.end(); it = next) {
		next = it;
		++next;

		int fd = (*it).first;
		FD_ENTRY *e = (*it).second;

		if (FD_ISSET(fd, &readfds))
			e->listener->onSocketRead();
		if (FD_ISSET(fd, &writefds))
			e->listener->onSocketWrite();
		if (FD_ISSET(fd, &exceptfds))
			e->listener->onSocketException();
	}

	return true;
}
