/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   copyright            : (C) 2003 by Zhang Yong                         *
 *   email                : z-yong163@163.com                              *
 ***************************************************************************/

#include "udpsession.h"
#include "socket.h"
#include "sha.h"
#include "icqlog.h"


#define SEQ_MASK	0x7fff

#define UDP_VER		1
#define TCP_VER		1

#define DEFAULT_ZEROK_SEQUENCES 500


static void readContactInfo(InPacket &in, CONTACT_INFO &info)
{
	in >> info.nick >> info.gender >> info.birth >> info.email;
	in >> info.country >> info.city >> info.address;
	in >> info.postcode >> info.tel >> info.mobile;
	in >> info.realname >> info.occupation >> info.homepage >> info.intro;
}


UDPSession::UDPSession(UDPSessionListener *l)
{
	listener = l;
	udpSocket = NULL;
	realIP = ourIP = 0;

	reset();
}

UDPSession::~UDPSession()
{
	clearSendQueue();

	if (udpSocket)
		delete udpSocket;
}

void UDPSession::connect(Socket *sock, const char *host, int port)
{
	if (udpSocket)
		delete udpSocket;

	udpSocket = sock;
	udpSocket->create(SOCK_DGRAM, this);
	udpSocket->selectEvent(Socket::READ | Socket::WRITE | Socket::EXCEPTION);
	udpSocket->connect(host, port);
}

void UDPSession::reset()
{
	numClients = 0;

	sessionID = ((rand() << 16) | (rand() & 0xffff));
	sendSeq = (rand() & SEQ_MASK);
	recvSeq = 0;
	memset(seqWindow, 0, sizeof(seqWindow));

	clearSendQueue();
}

void UDPSession::clearSendQueue()
{
	list<UDPOutPacket *>::iterator it;
	for (it = sendQueue.begin(); it != sendQueue.end(); ++it)
		delete *it;

	sendQueue.clear();
}

bool UDPSession::checkSeq(uint16 seq)
{
	uint8 &byte = seqWindow[seq / 8];
	uint8 mask = (1 << (seq % 8));
	if (byte & mask)
		return false;

	byte |= mask;
	return true;
}

void UDPSession::checkSendQueue()
{
	time_t current_time = time(NULL);

	while (!sendQueue.empty()) {
		UDPOutPacket *out = sendQueue.front();

		if (out->expire_time > current_time)
			break;

		sendQueue.pop_front();

		out->attempts++;
		if (out->attempts >= MAX_SEND_ATTEMPTS) {
			if (out->cmd == CMD_KEEPALIVE)
				listener->onConnect(false);
			else
				listener->onSendError(out->seq);

			delete out;

		} else {
			sendDirect(out);

			out->expire_time = current_time + SEND_TIMEOUT;
			sendQueue.push_back(out);
		}
	}
}

void UDPSession::createPacket(UDPOutPacket &out, uint16 cmd, uint16 seq)
{
	out.expire_time = time(NULL) + SEND_TIMEOUT;
	out.cmd = cmd;
	out.seq = seq;

	out << (uint16) UDP_VER << (uint32) 0;
	out << sessionID << seq << cmd;
}

UDPOutPacket *UDPSession::createPacket(uint16 cmd)
{
	UDPOutPacket *out = new UDPOutPacket;

	sendSeq = (sendSeq + 1) & SEQ_MASK;
	createPacket(*out, cmd, sendSeq);
	return out;
}

void UDPSession::sendPacket(UDPOutPacket *out)
{
	sendDirect(out);
	sendQueue.push_back(out);
}

void UDPSession::sendAck(uint16 seq)
{
	UDPOutPacket out;
	createPacket(out, CMD_ACK, seq);
	sendDirect(&out);
}

void UDPSession::sendDirect(UDPOutPacket *out)
{
	udpSocket->send(out->getData(), out->getSize());
}

void UDPSession::sendKeepAlive()
{
	UDPOutPacket *out = createPacket(CMD_KEEPALIVE);
	sendPacket(out);
}

void UDPSession::registerUser(const char *name, const char *passwd)
{
	ICQ_LOG("register %s...\n", name);
	reset();
	
	char token[11];
	char hash[41];
	char str[51];
	uint16 sequence = DEFAULT_ZEROK_SEQUENCES;
	
	/* generate new token */
	snprintf(token, 11, "%X", (unsigned int) time(NULL));
	
	shahash_r(passwd, hash);
	snprintf(str, 51, "%s%s", hash, token);
	shahash_r(str, hash);
	
	for(int i = 0; i < sequence; i++)
		shahash_r(hash, hash);
	
	hash[40] = '\0';

	
	UDPOutPacket *out = createPacket(CMD_REGISTER);
	*out << name << hash << token << sequence;
	sendPacket(out);
}

void UDPSession::preLogin(const char *name)
{
	reset();
	
	UDPOutPacket *out = createPacket(CMD_PRE_LOGIN);
	*out << name;
	sendPacket(out);
}

void UDPSession::login(const char *name, const char *passwd, const char *token, uint16 sequence, uint32 status, uint16 port)
{	
	char hash[41];
	char str[51];
		
	shahash_r(passwd, hash);
	snprintf(str, 51, "%s%s", hash, token);
	shahash_r(str, hash);
	
	int i;
	for(i = 1; i < sequence; i++)
		shahash_r(hash, hash);
	
	hash[40] = '\0';

	UDPOutPacket *out = createPacket(CMD_LOGIN);
	*out << name << hash;

	//need to update token.
	if (sequence == 1) {
		sequence = DEFAULT_ZEROK_SEQUENCES;
		
		char newToken[11];	
		snprintf(newToken, 11, "%X", (unsigned int) time(NULL));
	
		shahash_r(passwd, hash);
		snprintf(str, 51, "%s%s", hash, newToken);
		shahash_r(str, hash);
	
		for(i = 0; i < sequence; i++)
			shahash_r(hash, hash);
	
		hash[40] = '\0';
		
		*out << hash << newToken << sequence;
	}

	*out << status << (uint16) TCP_VER << realIP << port;
	

	sendPacket(out);
}

void UDPSession::logout()
{
	UDPOutPacket out;
	createPacket(out, CMD_LOGOUT, ++sendSeq);
	sendDirect(&out);
}

void UDPSession::changeStatus(uint32 status)
{
	UDPOutPacket *out = createPacket(CMD_CHANGE_STATUS);
	*out << status;
	sendPacket(out);
}

void UDPSession::sendMessage(uint8 type, const char *to, const char *text)
{
	UDPOutPacket *out = createPacket(CMD_MESSAGE);
	*out << to << type << text;
	sendPacket(out);
}

void UDPSession::searchRandom()
{
	UDPOutPacket *out = createPacket(CMD_SEARCH_RANDOM);
	sendPacket(out);
}

void UDPSession::searchUser(const char *name, const char *nick, const char *email)
{
	UDPOutPacket *out = createPacket(CMD_SEARCH);
	*out << name << nick << email;
	sendPacket(out);
}

void UDPSession::addContact(const char *name)
{
	UDPOutPacket *out = createPacket(CMD_ADD_CONTACT);
	*out << name;
	sendPacket(out);
}

void UDPSession::delContact(const char *name)
{
	UDPOutPacket *out = createPacket(CMD_DEL_CONTACT);
	*out << name;
	sendPacket(out);
}

void UDPSession::getContactList()
{
	UDPOutPacket *out = createPacket(CMD_GET_CONTACT_LIST);
	sendPacket(out);
}

void UDPSession::getContactInfo(const char *name)
{
	UDPOutPacket *out = createPacket(CMD_GET_CONTACT_INFO);
	*out << name;
	sendPacket(out);
}

void UDPSession::getUserInfo()
{
	UDPOutPacket *out = createPacket(CMD_GET_USER_INFO);
	sendPacket(out);
}

void UDPSession::updateUserInfo(USER_INFO &info)
{
	UDPOutPacket *out = createPacket(CMD_UPDATE_USER_INFO);

	*out << info.auth << info.nick << info.gender << info.birth << info.email;
	*out << info.country << info.city << info.address;
	*out << info.postcode << info.tel << info.mobile;
	*out << info.realname << info.occupation << info.homepage << info.intro;

	sendPacket(out);
}

void UDPSession::preChangePassword()
{
	UDPOutPacket *out = createPacket(CMD_PRE_CHANGE_PASSWD);
	sendPacket(out);
}

void UDPSession::changePassword(const char *oldPasswd, const char *token, uint16 sequence, const char *newPasswd)
{
	char hash[41];
	char str[51];
		
	shahash_r(oldPasswd, hash);
	snprintf(str, 51, "%s%s", hash, token);
	shahash_r(str, hash);
	
	int i;
	for(i = 1; i < sequence; i++)
		shahash_r(hash, hash);
	
	hash[40] = '\0';

	UDPOutPacket *out = createPacket(CMD_CHANGE_PASSWD);
	*out << hash;
	
	sequence = DEFAULT_ZEROK_SEQUENCES;
		
	char newToken[11];	
	snprintf(newToken, 11, "%X", (unsigned int) time(NULL));
	
	shahash_r(newPasswd, hash);
	snprintf(str, 51, "%s%s", hash, newToken);
	shahash_r(str, hash);
	
	for(i = 0; i < sequence; i++)
		shahash_r(hash, hash);
	
	hash[40] = '\0';
		
	*out << hash << newToken << sequence;
	
	sendPacket(out);
}

void UDPSession::onSocketRead()
{
	char buf[MAX_PACKET_SIZE];

	int n = udpSocket->receive(buf, sizeof(buf));
	if (n < (int) sizeof(UDP_PACKET_HDR))
		return;

	UDPInPacket in(buf, n);
	onPacketReceived(in);
}

void UDPSession::onSocketWrite()
{
	udpSocket->selectEvent(Socket::READ | Socket::EXCEPTION);

	int fd = udpSocket->getFd();
	sockaddr_in addr;
	socklen_t len = sizeof(addr);
	getsockname(fd, (sockaddr *) &addr, &len);

	realIP = ntohl(addr.sin_addr.s_addr);

	ICQ_LOG("my realIP is %s\n", inet_ntoa(addr.sin_addr));

	listener->onConnect(true);
}

void UDPSession::onSocketException()
{
	listener->onConnect(false);
}

void UDPSession::onPacketReceived(UDPInPacket &in)
{
	if (sessionID != in.header.sid) {
		ICQ_LOG("session id does not match(sid = %08x, cmd = %d).\n",
			in.header.sid, in.header.cmd);
		return;
	}

	uint16 seq = in.header.seq;
	uint16 cmd = in.header.cmd;

	if (cmd == CMD_ACK) {
		onAck(in);
		return;
	}

	sendAck(seq);

	if (!checkSeq(seq)) {
		ICQ_LOG("packet %d is ignored.\n", seq);
		return;
	}

	switch (cmd) {
	case CMD_REGISTER:
		onRegisterReply(in);
		break;

	case CMD_PRE_LOGIN:
		onPreLoginReply(in);
		break;
	
	case CMD_LOGIN:
		onLoginReply(in);
		break;

	case CMD_KEEPALIVE:
		onKeepAliveReply(in);
		break;

	case CMD_ADD_CONTACT:
		onAddContactReply(in);
		break;

	case CMD_GET_CONTACT_LIST:
		onContactListReply(in);
		break;

	case CMD_SRV_USER_ONLINE:
		onUserOnline(in);
		break;

	case CMD_SRV_USER_OFFLINE:
		onUserOffline(in);
		break;

	case CMD_SRV_USER_STATUS:
		onUserStatus(in);
		break;

	case CMD_SRV_SEARCH_RESULT:
		onSearchResult(in);
		break;

	case CMD_SRV_MESSAGE:
		onRecvMessage(in);
		break;

	case CMD_GET_CONTACT_INFO:
		onContactInfoReply(in);
		break;

	case CMD_GET_USER_INFO:
		onUserInfoReply(in);
		break;
	
	case CMD_PRE_CHANGE_PASSWD:
		onPreChangePasswordReply(in);
		break;
	}
}

void UDPSession::onAck(UDPInPacket &in)
{
	uint16 seq = in.header.seq;

	list<UDPOutPacket *>::iterator it;

	for (it = sendQueue.begin(); it != sendQueue.end(); ++it) {
		UDPOutPacket *p = *it;
		if (p->seq == seq) {
			ICQ_LOG("packet %d is acked.\n", seq);

			in >> numClients;
			listener->onAck(seq);

			sendQueue.erase(it);
			delete p;
			return;
		}
	}

	ICQ_LOG("Ack %d is ignored.\n", seq);
}

void UDPSession::onRegisterReply(UDPInPacket &in)
{
	uint8 error;
	in >> error;

	listener->onRegisterReply(error);
}

void UDPSession::onPreLoginReply(UDPInPacket &in)
{
	uint16 sequence;
	const char *token;

	in >> sequence;
	if (sequence > 0) //sequence equal to 0 means invalid user.
		in >> token;
	else
		token = NULL;

	listener->onPreLoginReply(sequence, token);
}

void UDPSession::onLoginReply(UDPInPacket &in)
{
	uint8 error;
	in >> error >> ourIP;

	in_addr addr;
	addr.s_addr = htonl(ourIP);
	ICQ_LOG("ourIP is %s.\n", inet_ntoa(addr));

	listener->onLoginReply(error);
}

void UDPSession::onKeepAliveReply(UDPInPacket &in)
{
	in >> numClients;
}

void UDPSession::onRecvMessage(UDPInPacket &in)
{
	uint8 type;
	uint32 when;
	const char *from, *text;

	in >> from >> when >> type >> text;

	listener->onRecvMessage(type, from, (time_t) when, text);
}

void UDPSession::onUserOnline(UDPInPacket &in)
{
	ONLINE_INFO info;

	in >> info.name >> info.status;
	in >> info.tcp_ver >> info.ip >> info.real_ip >> info.msg_port;

	listener->onUserOnline(info);
}

void UDPSession::onUserOffline(UDPInPacket &in)
{
	const char *name;
	in >> name;

	listener->onUserOffline(name);
}

void UDPSession::onUserStatus(UDPInPacket &in)
{
	const char *name;
	uint32 status;

	in >> name >> status;

	listener->onUserStatus(name, status);
}

void UDPSession::onSearchResult(UDPInPacket &in)
{
	uint16 n;
	in >> n;

	SEARCH_RESULT *result = new SEARCH_RESULT[n];

	for (int i = 0; i < n; i++) {
		SEARCH_RESULT *p = result + i;

		in >> p->name >> p->status >> p->nick;
		in >> p->auth >> p->gender >> p->age;
	}

	listener->onSearchResult(result, n);

	delete []result;
}

void UDPSession::onAddContactReply(UDPInPacket &in)
{
	const char *name;
	uint8 auth;

	in >> name >> auth;

	listener->onAddContactReply(name, auth);
}

void UDPSession::onContactListReply(UDPInPacket &in)
{
	uint16 n;
	in >> n;

	const char **list = new const char *[n];

	for (int i = 0; i < n; i++)
		in >> list[i];

	listener->onContactListReply(list, n);
	delete []list;
}

void UDPSession::onContactInfoReply(UDPInPacket &in)
{
	CONTACT_INFO c;

	in >> c.name;
	readContactInfo(in, c);

	listener->onContactInfoReply(c);
}

void UDPSession::onUserInfoReply(UDPInPacket &in)
{
	USER_INFO user;
	user.name = NULL;

	in >> user.auth;
	readContactInfo(in, user);

	listener->onUserInfoReply(user);
}

void UDPSession::onPreChangePasswordReply(UDPInPacket &in)
{
	uint16 sequence;
	const char *token;

	in >> sequence;
	if (sequence > 0) //sequence equal to 0 means unknow error.
		in >> token;
	else
		token = NULL;

	listener->onPreChangePasswordReply(sequence, token);
}
